import random
from typing import Union, Iterable
import re
from abc import ABC

from src.movement import Movement, list_moves_to_list_str
from src.enums import LearnStrat


class Player(ABC):
    """Abstract Base Class for Players."""

    turn: int
    name: str

    def __init__(self, first: bool, name: str):
        if first: self.turn = 1
        else: self.turn = 0
        self.name = name

    def make_move(self) -> str: ...

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.name

class PlayerHuman(Player):
    """Applicable Human Player class.

    Attr:
        first (bool): If this player goes first or not.
        name (str): Name of the player"""

    def make_move(self) -> Movement:
        """Requests player to input moves.

        Returns:
            Movement: Movement object of player input
        """
        while True:
            start_position = input("Which piece do you wanna move?: ")
            end_position = input("Where do you wanna move the piece?: ")
            if re.match(
                "^[a-zA-Z][0-9].[a-zA-Z][0-9]$",
                start_position + "-" + end_position
                ):
                return Movement.from_str(start_position + "-" + end_position)
            else:
                print("Wrong input... Please refer to actual coordinates of the board!")

class RandomComp(Player):
    """Automatic player which performes random move.

    Attr:
        first (bool): If this player goes first or not.
        name (str): Name of the player"""

    def make_move(self, legal_moves: Iterable[Movement]) -> Movement:
        """Lets the computer perform a random move.

        Args:
            possible_moves (Iterable[Movement]): List of Moves to choose from.

        Returns:
            Movement: Return random Movement object from list.
        """

        try:
            move = random.choice(legal_moves)
        except IndexError:
            return False

        return move

class AIComp(Player):# TODO: get method to check on what boards ai was trained
    """AI driven computer player.
    Keeps on learning while playing based on given strategy input.

    Attr:
        first (bool): If this player goes first or not.
        name (str): Name of the player
        memory_table (dict[str:str]): dynamically created dict with all boards seen and corresponding moves available.
        last_move (Movement): Stores the last move performed for learning purposes"""

    memory_table: dict[str: str]
    last_move: tuple()

    def __init__(self, first: bool, name: str, strategy: int):
        """AI based player. Allows the usage of basic punishment and reward strategies.
        Builds it's own memory table and doesn't require any hardcoding.

        Args:
            first (bool): State if the player is first to go.
            strategy (int): int representation of the strategy to use.
                            Please use the provided Enum class 'LearnStrat'
        """

        super().__init__(first, name)

        self.memory_table = {}
        self.last_move = ("","")

        self._def_strat(strategy)

    def _def_strat(self, strat: int):
        """Defines strategy to use. According method is asigned to method 'feedback'.

        Args:
            strat (int): Please use the LearningStrat Enum for asignment.

        Raises:
            ValueError: Raised when input is not asigned to any given strategy.
        """

        if strat == LearnStrat.REWARD:
            self.feedback = self._reward
        elif strat == LearnStrat.PUNISH:
            self.feedback = self._punishment
        elif strat == LearnStrat.BOTH:
            self.feedback = self._reward_and_punishement

        else: raise ValueError("Given Int for Strategy not in range! Please use 'LearnStrat' class")

    def _add_board_to_memory(self, possible_moves: Iterable[Movement],
                             board_id: str):
        """Adds possible moves and board_id to memory table.

        Args:
            possible_moves (Iterable[Movement]): Iterable of all possible moves for given board_id
            board_id (str): string presentation of a board
        """

        self.memory_table[board_id] = list_moves_to_list_str(possible_moves)

    def make_move(self, legal_moves: Iterable[Movement],
                  board_id: str) -> Union[Movement, bool]:
        """Lets the player make a move.
        Move is based on memory table.
        If board_id is not in memory_table, information is added.

        Args:
            possible_moves (Iterable[Movement]): all possible moves for board_id
            board_id (str): string presentation of given board

        Returns:
            Union[Movement, bool]: Returns Movement or False if no move is available.
        """

        if board_id not in self.memory_table:
            self._add_board_to_memory(legal_moves, board_id)

        try:
            self.last_move = (board_id,
                              random.choice(self.memory_table[board_id]))
        except IndexError:
            return False # entspricht aufgeben

        return Movement.from_str(self.last_move[1])

    def _punishment(self, victory: bool):
        """Punishment strategy for AI.
        Whenever a move leads to losing, it is removed from memory table.

        Args:
            victory (bool): If AIComp has won or not.
        """

        if not victory:
            self.memory_table[self.last_move[0]].remove(self.last_move[1])

    def _reward(self, victory: bool):
        """Rewarding strategy for AI.
        Whenever a move leads to winning, another copy is added to memory table.

        Args:
            victory (bool): If AIComp has won or not.
        """

        if victory:
            self.memory_table[self.last_move[0]].append(self.last_move[1])

    def _reward_and_punishement(self, victory: bool):
        """Punishment and Rewarding strategy combined simultaniously.

        Args:
            victory (bool): If AIComp has won or not.
        """

        self._reward(victory)
        self._punishment(victory)
