import numpy as np
from string import ascii_uppercase

from src.piece import BasePiece
from src.movement import Movement, check_move_in_list
from src.enums import PlayerOrder


class Board:
    """Class to represent chessbaords.
    Generates necessary pieces of given type.

    Args:
        piece_used (BasePiece): Piece to use for the game.
        size (int): Size of the chessboard. used for length and width.

    Attr:
        size (int): length of the chessboard with area "size x size".
        board (np.ndarray): Array representation of the board.
        piece_used (BasePiece): Pieces used for a game.
        legal_move (List[Movement]): a list of all legal moves for the given board.
    """

    size: int
    board: np.ndarray
    piece_used = BasePiece
    legal_moves: list[Movement]

    def __init__(self, piece_used: BasePiece, size: int = 3):
        self.size = size
        self.piece_used = piece_used

        self.initialize_board()

    def _initial_board(self):
        """Builds the starting position with fixed values."""

        board = [self.piece_used(0,i,PlayerOrder.PLAYER1) for i in range(self.size)]

        for _ in range(2, self.size):
            board.extend([0 for _ in range(self.size)])

        board.extend([self.piece_used(self.size-1,i,PlayerOrder.PLAYER2)
                    for i in range(self.size)])

        self.board = np.array(board).reshape(self.size, self.size)

    def _update_legal_moves(self, player: PlayerOrder):
        """Updates the 'legal_moves' attribute.
        Only Moves for the given player are considered.

        Args:
            player (int): Player that has to do his turn. Used Enum PlayerOrder!
        """
        self.legal_moves = []
        for position in self.board.reshape(-1):
            if not position: continue
            if position.player != player: continue

            piece_moves = position.get_all_moves(self.board)
            self.legal_moves.extend(piece_moves)

    def initialize_board(self):
        """Initializes the chessbord with the standard starting position.
        Can be used to reset the board.
        """
        self._initial_board()
        self._update_legal_moves(player = PlayerOrder.PLAYER1)

    def get_id(self) -> str:
        """Get an ID representation of the board.

        Returns:
            str: ID of the current board.
        """
        out = ""
        for idx, board_cell in enumerate(self.board.reshape(-1)):
            if board_cell:
                out += f"{idx // self.size}{idx % self.size}{board_cell.get_id()}"
            else:
                out += f"{idx // self.size}{idx % self.size}X0"
        return out

    def _move_piece(self, move: Movement):
        """Move a piece on the board.
        Updates the position of the piece as well.

        Args:
            move (Movement): Move to make.
        """
        self.board[move.start].move_piece(move)
        self.board[move.end] = self.board[move.start]
        self.board[move.start] = 0

    def apply_move(self, move: Movement, player: PlayerOrder) -> bool:
        """Apply a given move to the board.
        Checks if the move is legal.

        Args:
            move (Movement): Move that should be made.
            player (int): Player that wants to make the move.

        Returns:
            bool: If the move was legal and was applied or not.
        """
        if not check_move_in_list(move, self.legal_moves):
            return False

        self._move_piece(move)
        self._update_legal_moves(player)
        return True

    def check_win(self, current_player: PlayerOrder) -> PlayerOrder: #TODO:
        """Checks if the board state is a win for any player.

        Args:
            turn (int): Turnnumber of the game. Necessary to calculate winning player if no legal moves are possible.

        Returns:
            PlayerOrder: Returns the corresponding Enum value
        """
        if not self.legal_moves:
            if current_player == PlayerOrder.PLAYER1:
                return PlayerOrder.PLAYER1
            else: return PlayerOrder.PLAYER2

        for piece in self.board[0]:
            if not piece: continue
            if piece.player == PlayerOrder.PLAYER2: return PlayerOrder.PLAYER2

        for piece in self.board[-1]:
            if not piece: continue
            if piece.player == PlayerOrder.PLAYER1: return PlayerOrder.PLAYER1

        return PlayerOrder.NOPLAYER

    def __str__(self):
        out = "\n"

        for idx, sub_array in enumerate(self.board):
            out += (f"{idx} || "
                    + "   ".join([str(position) for position in sub_array])
                    + "\n")

        out += f"  ==={'='.join(['' for _ in range(len(self.board[0])*4)])}\n"

        out += "     " + "   ".join([
            ascii_uppercase[val] for val in range(len(self.board[0]))
            ])

        return out