#%% Shreksapawn - a learning game based on hexapawn
# My version contains two different levels
# Level 1: Non-learning system - computer selects randomly one of the allowed moves
# Level 2: Learning system I (learning by punishment) 
# Level 3: Learning system II (learning by punishment and reward) - system learns faster as in level 2,
# but needs more games to reach perfection
# Level 4: Expert - the computer will always wins as it only randomly select of move, 
# which will led to victory [NOT IMPLEMENTED]
# -----------------------------------------------------------------------
# The programme is not perfect, you can improve it further e.g. by
# build a graphical user interface
# name the pawns like figures in Shrek
# give the user the opportunity to name its team, to select symbols for the team, ...
# extend the board size 
# implement level 4 (e.g. check for L2 and L3 if computer already reached perfection)
#%% Import libaries
import random
import os
#%% Define several classes for the game
class Board():

    def __init__(self,size:int):
        """ Set board size and initalise empty fields"""
        self.size = size
        self.fields = []
        for i in range(size*size):
            self.fields.append(' ') 

    def show_board(self):
        """ print board row by row"""
        row_1 = '\n'
        for i in range (self.size):
            row_1 += '  X' + str(i+1)
        print(row_1)
        for i in range(self.size):
            row = 'Y' + str(i+1)
            for j in range(self.size):
                row += ' ' + self.fields[self.size*i+j] + '  '
            print(row)
            
    def set_fields(self,X1:list,Y1:list,symbol1:str,X2:list,Y2:list,symbol2:str):
        """ Set fields of the board either as empty or with the symbols of the two players"""
        for i in range(self.size**2):
            self.fields[i] = ' '
        for i in range(len(X1)):
            if X1[i] > -1:
                self.fields[X1[i]+Y1[i]*self.size] = symbol1
            if X2[i] > -1:
                self.fields[X2[i]+Y2[i]*self.size] = symbol2  
             
class Team():
    def __init__(self,n_pawns:int,p_type:str):
        """ Set the position, direction and symbol of the two teams
        Team User has a X and starts at the bottom of the board, 
        The second team has an O and starts at the top of the board"""
        self.posX = []
        self.posY = []
        self.p_type = p_type
        for i in range(n_pawns):
            self.posX.append(i)
            if p_type == 'User':
                self.symbol = 'X'
                self.posY.append(n_pawns-1)
                self.direction = -1
            else:
                self.symbol = 'O'
                self.posY.append(0)
                self.direction = 1

    def move_pawn(self,index:int, X:int, Y:int):
        """ Move pawn of given index on the new position X and Y"""
        self.posX[index] = X
        self.posY[index] = Y

class Moves():
    def __init__(self):
        """ Initialise move_list as empty list"""
        self.move_list = []

    def is_in_moveList(self,X_user:list, Y_user:list, X_comp:list, Y_comp:list)->int:
        """Function checks if current state is already in list,
           returns index or -1, if not in list """
        for index, value in enumerate(self.move_list):
            X1 = self.move_list[index]['Position']['X_user']
            Y1 = self.move_list[index]['Position']['Y_user']
            X2 = self.move_list[index]['Position']['X_comp']
            Y2 = self.move_list[index]['Position']['Y_comp']
            if (X1 == X_user) and (Y1 == Y_user) and (X2 == X_comp) and (Y2 == Y_comp):
                return index
        return -1    
    
    def append_moveList(self, X_user:list, Y_user:list, X_comp:list, Y_comp:list, poss_moves:list)->int:
        """Function appends move_list if current state is not in list"""
        entry = {'Position':{'X_user':X_user, 'Y_user':Y_user, 'X_comp':X_comp,'Y_comp':Y_comp},'Moves':poss_moves}
        self.move_list.append(entry)
        return len(self.move_list)-1

    def remove_move(self, index_list:int, index_move:int):
        """Remove move from move_list (punishment situation)""" 
        self.move_list[index_list]['Moves'].remove(index_move)

    def add_move(self, index_list:int, move:dict):
        """Add move to move_list (reward situation)"""
        self.move_list[index_list]['Moves'].append(move)
   
    def calculate_allowed_moves(self, size:int, X_p1:list, Y_p1:list, X_p2:list, Y_p2:list,direction:int)->list:
        """ Calulate all possible moves from the given position of team 1 and team 2
        and return the moves as list"""
        self.poss_moves = []
        for pawn in range(len(X_p1)):
            if (X_p1[pawn]) != -1:
                if self.check_move(size, X_p1[pawn],Y_p1[pawn]+direction,X_p2,Y_p2) == 2:
                    # check if you can go forward
                    move= {'Pawn_index': pawn,'NewX':X_p1[pawn],'NewY':Y_p1[pawn]+direction, 'OldX':X_p1[pawn], 'OldY':Y_p1[pawn]}
                    self.poss_moves.append(move)
                if self.check_move(size, X_p1[pawn]-1,Y_p1[pawn]+direction,X_p2,Y_p2) == 1:
                    # check if you can go left-diagonal
                    move= {'Pawn_index':pawn,'NewX':X_p1[pawn]-1,'NewY':Y_p1[pawn]+direction, 'OldX':X_p1[pawn], 'OldY':Y_p1[pawn]}
                    self.poss_moves.append(move)
                if self.check_move(size, X_p1[pawn]+1,Y_p1[pawn]+direction,X_p2,Y_p2) == 1:
                    # check if you can go right-diagonal
                    move= {'Pawn_index':pawn,'NewX':X_p1[pawn]+1,'NewY':Y_p1[pawn]+direction, 'OldX':X_p1[pawn], 'OldY':Y_p1[pawn]}
                    self.poss_moves.append(move)  
        return self.poss_moves                                 
    
    def check_move(self,size:int, X:int,Y:int,X_p2:list,Y_p2:list)->int:
        """ Checks if move is possible"""
        if (X < 0) or (Y < 0) or (X >= size) or (Y >= size):
            # out of the board
            return -1
        for i in range(len(X_p2)):
            # new position is blocked by opponent pawn
            if X == X_p2[i] and Y == Y_p2[i]:
                return 1
        # new position is free    
        return 2
    
    def print_moves(self,poss_moves:list):
        """ print given moves from list on screen"""
        print('Possible moves:')
        for i in range(len(poss_moves)):
            print(str(i+1) +': Pawn on X'+str(poss_moves[i]['OldX']+1) +'/Y'+str(poss_moves[i]['OldY']+1), 'to X'+str(poss_moves[i]['NewX']+1)+'/Y'+str(poss_moves[i]['NewY']+1))        
            
class Game():
    def __init__(self, size:int):
        """ Initalise all important parts for the game as Board, Moves, level and size"""
        self.level = 2
        self.size = size
        self.board = Board(self.size)
        self.moves = Moves()
        self.counter = 0
        self.history = []

    def play(self)->list:
        """ Starts a round of the game"""
        self.counter += 1
        self.nround = 0
        if (self.counter == 1):
            # create Team 1 (User) and Team 2 (Computer)
            self.select_levels()
            self.team1 = Team(self.size,'User')
            self.team2 = Team(self.size,'Computer')
        else:
            # set all pawns in the starting position on the board
            for i in range(self.size):
                self.team1.move_pawn(i,i,self.size-1)
                self.team2.move_pawn(i,i,0)
        # Display board and set active and opponent player    
        self.board.set_fields(self.team1.posX, self.team1.posY, self.team1.symbol, self.team1.posX, self.team2.posY, self.team2.symbol)
        self.board.show_board()
        while (self.is_game_over(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY,self.team1.direction, 0) == 2):
            self.nround +=1
            if (self.nround % 2):
                self.active = self.team1
                self.opponent = self.team2
            else:
                self.active = self.team2
                self.opponent = self.team1       
           
           # Select next move
            if self.active.p_type == 'User' or self.level == 1 or self.size !=3:
                # Calculate all possible moves for user, board-size != 3 or for Level 1
                self.poss_moves = self.moves.calculate_allowed_moves(self.size,self.active.posX,self.active.posY,self.opponent.posX,self.opponent.posY,self.active.direction)
                if (self.poss_moves == []):
                    print(self.active.p_type+' cannot make a valid move.')
                    if self.active.p_type == 'User':
                        self.print_result(0)
                    else:
                        self.print_result(1)
                    return self.history
                elif self.active.p_type == 'User':
                    self.moves.print_moves(self.poss_moves)
                    next_move = int(input('Please select your next move: ')) -1
                    if (next_move not in list(range(len(self.poss_moves)))):
                        print('Invalid input. Move 1 will be performed.')
                        next_move = 0
                else:
                    next_move = random.randrange(len(self.poss_moves))
            elif self.level == 2 or self.level == 3:
                if self.counter == 1:
                    self.poss_moves = self.moves.calculate_allowed_moves(self.size,self.active.posX,self.active.posY,self.opponent.posX,self.opponent.posY,self.active.direction)
                    index_list = self.moves.append_moveList(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY,self.poss_moves)
                else:
                    index_list = self.moves.is_in_moveList(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY)
                    if index_list == -1:
                        self.poss_moves = self.moves.calculate_allowed_moves(self.size,self.active.posX,self.active.posY,self.opponent.posX,self.opponent.posY,self.active.direction)
                        index_list = self.moves.append_moveList(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY,self.poss_moves)
                    else:
                        self.poss_moves = self.moves.move_list[index_list]['Moves']
                if self.poss_moves == []:
                    self.print_result(1)
                    return self.history     
                next_move = random.randrange(len(self.poss_moves))
            # Make next move
            pawn = self.poss_moves[next_move]['Pawn_index']
            newX = self.poss_moves[next_move]['NewX']
            newY = self.poss_moves[next_move]['NewY']
            self.active.move_pawn(pawn, newX, newY)
            for i in range(self.size):
                if (self.opponent.posX[i] == newX and self.opponent.posY[i] == newY):
                    self.opponent.move_pawn(i, -1, -1)
            # Set fields on the board and display board
            self.board.set_fields(self.active.posX, self.active.posY, self.active.symbol, self.opponent.posX, self.opponent.posY,self.opponent.symbol)    
            self.board.show_board()
            # Control if game is won or lost for punishment or reward
            if (self.level == 2 or self.level == 3) and self.active.p_type == 'Computer':
                value = self.is_game_over(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY,self.team1.direction, 1)
                if value == 1:
                    # user won, punish system by removing last move from list
                    self.moves.remove_move(index_list,next_move)
                elif value == 0 and self.level == 3:
                    # computer won, reward
                    self.moves.add_move(index_list,self.poss_moves[next_move])

            
        winner = self.is_game_over(self.team1.posX,self.team1.posY,self.team2.posX,self.team2.posY,self.team1.direction, 0) 
        self.print_result(winner)    
        return self.history
    
    def print_result(self, winner:int):
        """ Print result on screen and append number to play history"""
        if winner == 0:
            print('Sorry. You lost.')
            self.history.append(0)
        else:
            print('Congratulations, you won.')    
            self.history.append(1)

    def select_levels(self):
        """ User can select a level from the list"""
        print('Please select a level:')
        print('Level 1 - Non-learning system: The computer selects randomly one of the allowed moves')
        print('Level 2 - Learning system (by punishment)')
        print('Level 3 - Learning system II (by punishment and reward)')
        user_level = int(input('Your choice: '))
        if ((user_level) in [1, 2, 3]):
            self.level = user_level
        else:
            print('Invalid input. The game starts on level 2.')

    def is_game_over(self, X_user:list, Y_user:list, X_comp:list, Y_comp:list,d_user:int,check_userMove:int)->int:
        """Check if one of three win situations is reached and returns
        0: computer wins
        1: user wins
        2: game is not over"""
        # Check if user can make a move
        if (check_userMove == 1):
            self.poss_moves = self.moves.calculate_allowed_moves(self.size,X_user,Y_user,X_comp,Y_comp,d_user)
            if self.poss_moves == []:
                return 0
        # Check if one player reached the other side
        for i in range(len(Y_user)):
            if (Y_user[i] == 0):
                return 1
            elif (Y_comp[i] == self.size-1):
                return 0
        # Check if one player has lost all figures
        if sum(X_user) == -1*len(X_user):
            return 0
        elif sum(X_comp) == -1*len(X_comp):
            return 1
        # If no one wins, return 0
        return 2

#%%
# Main programme
print('Welcome to my version of Shreksapawn - the game that learns (based on the game Hexapawn)')
start_game = input('Do you want to play some rounds to test how long you can win against the game? (y/n) ')
game = Game(3)
play = True
if (start_game == 'n'):
    exit_game = input('Are you sure to exit the game? (y/n) ')
    if (exit_game == 'y'):
        print('Exit Shreksapawn.')
        play = False
        quit()

while (play):
    os.system('cls')
    win_history = game.play()
    continue_game = input('Do you want to continue and start a new round? (y/n) ' )
    if continue_game != 'y':
        play = False

print('You won', sum(win_history), 'of', len(win_history), 'games.')
print ('Win history:',win_history)
