from __future__ import annotations
import re
from typing import Iterable

import src.coordinate_functions as util

def check_move_in_list(move: Movement, list_of_moves: Iterable[Movement]
                       ) -> bool:
    """Checks if a Movement instance is equivalent any other instance in a list.
    Movement instances do not have to be the same.
    Str representations are used for comparison.

    Args:
        move (Movement): Movement object to compare
        list_of_moves (Iterable[Movement]): list of Movements to check if it contains requested Movement

    Returns:
        bool: If move input has an equivalent in given List or not
    """
    for move_of_list in list_of_moves:
        if str(move) == str(move_of_list): return True

    return False

def list_moves_to_list_str(list_moves: Iterable[Movement]) -> list[str]:
    """Converts list of Movement objects to a list string representations.

    Args:
        list_moves (Iterable[Movement]): List to convert

    Returns:
        list[str]: List of string representations of the input.
    """
    out = []

    for move in list_moves:
        out.append(str(move))

    return out

class Movement:
    """Class representing movements of a chessboard.
    Start and endpoint can be accessed as matrix or coordinate presentation.
    String representations of complete moves are available as well."""

    start_row: int
    start_col: int
    end_row: int
    end_col: int

    def __init__(self, start_row: int, start_col: int,
                 end_row: int, end_col: int):
        self.start_row = start_row
        self.start_col = start_col
        self.end_row = end_row
        self.end_col = end_col

    @classmethod
    def from_str(cls, move: str) -> Movement:
        """Generates Movement object from a string.

        Args:
            move (str): Pattern '^[a-zA-Z][0-9].[a-zA-Z][0-9]$'

        Raises:
            ValueError: When move is not of given regex pattern

        Returns:
            Movement: Corresponding Movement object
        """
        if not re.match("^[a-zA-Z][0-9].[a-zA-Z][0-9]$", move):
            raise ValueError("The given string is not of format '^[a-zA-Z][0-9].[a-zA-Z][0-9]$'")

        start_coord = util.coord_to_matrix(move[0:2])
        end_coord = util.coord_to_matrix(move[-2:])

        return cls(
            start_coord[0],
            start_coord[1],
            end_coord[0],
            end_coord[1]
        )

    @property
    def start(self) -> tuple[int, int]:
        """Start point in matrix form.

        Returns:
            tuple[int, int]: Row and Col matrix, respectively
        """
        return self.start_row, self.start_col

    @property
    def end(self) -> tuple[int, int]:
        """End point in matrix form.

        Returns:
            tuple[int, int]: Row and Col matrix, respectively
        """
        return self.end_row, self.end_col

    @property
    def start_coord(self) -> str:
        """Start point in coordinate form.

        Returns:
            tuple[int, int]: Row and Col matrix, respectively
        """
        return util.matrix_to_coord(self.start_row, self.start_col)

    @property
    def end_coord(self) -> str:
        """End point in coordinate form.

        Returns:
            tuple[int, int]: Row and Col matrix, respectively
        """
        return util.matrix_to_coord(self.end_row, self.end_col)

    def __str__(self) -> str:
        return f"{util.matrix_to_coord(self.start_row, self.start_col)}-{util.matrix_to_coord(self.end_row, self.end_col)}"
