from typing import Union

from src.enums import PlayerOrder
from src.player import Player, AIComp, RandomComp
import src.tui as tui
from src.board import Board
from src.movement import Movement


def train_ai(iterations: int, board: Board, player_ai: AIComp,
             player_enemy: Union[RandomComp, AIComp]):
    """Train an AI Player in a seperate game.
    Any enemy can be set to any computer

    Args:
        iterations (int): amount of training games to play.
        board (Board): Board to play on.
        player_ai (AIComp): The player to be trained.
        player_enemy (Union[RandomComp, AIComp]): Trainings partner of the ai. This player always goes first!
    """
    for _ in range(iterations):
        Game(board, player_enemy, player_ai, False).game()

class Game:
    """Game class to create and play games of Hexapawn.

    Attr:
        board (Board): The board to play on
        player_one (Player): Any Player of the abstract base class
        player_two (Player): Any Player of the abstract base class
        players (tuple[Player, Player]): Dynamically created. Used to determine active player basd on turn number.
        print_gamestates (bool): Whether this game should use print functions or not.
    """

    board: Board
    player_one: Player
    player_two: Player
    players: tuple[Player, Player]
    print_gamestates: bool

    def __init__(self, board: Board, player_one: Player,
                 player_two: Player, print_gamestates: bool = True):
        self.board = board
        self.player_one = player_one
        self.player_two = player_two
        self.players = (player_two, player_one)
        self.print_gamestates = print_gamestates

        self.scoreboard = {self.player_two: 0, self.player_one: 0}

    def _player_make_move(self, player: Player) -> Movement:
        """Requests given player to make a move.
        Handles different Player classes with different argument requirements.

        Args:
            player (Player): Player to make a move.

        Returns:
            Movement: Return the move to make.
        """
        if isinstance(player, AIComp):
            return player.make_move(
                self.board.legal_moves, self.board.get_id())

        elif isinstance(player, RandomComp):
            return player.make_move(self.board.legal_moves)

        else:
            return player.make_move()

    def _eval_game(self, winning_player: PlayerOrder):
        """Evaluates who is the winner and updates ai, scoreboard and human in front of computer.

        Args:
            winning_player (int): int presentation of player
        """
        if winning_player == PlayerOrder.PLAYER1:
            self._feedback_if_ai(self.player_two, False)
            self.scoreboard[self.player_one] += 1
            if self.print_gamestates: print("You won, congrats!")

        else:
            self._feedback_if_ai(self.player_two, True)
            self.scoreboard[self.player_two] += 1
            if self.print_gamestates: print("You lost. How unfortunate...")

    @staticmethod
    def _feedback_if_ai(player: Player, victory: bool):
        """If player is instance of AIComp, the feedback method is called.

        Args:
            player (Player): Any
            victory (bool): Whether given player has won the game or not.
        """
        if isinstance(player, AIComp):
            player.feedback(victory)

    def _player_turn(self, active_player: Player, turn: int
                     ) -> Union[Movement, bool]:
        """Calling Player_make_move and checking output for legality and surrender.

        Args:
            active_player (Player): Any
            turn (int): Turn that is played. Necessary to update board. #TODO: could be improved for more stand alone

        Returns:
            Union(Movement, bool): Return either a legal move, or False if a player gives up
        """
        for _ in range(5):

            move = self._player_make_move(active_player)

            if move == False:
                if self.print_gamestates :
                    print(f"{active_player} has surrendered. You win!")
                return False

            if self.board.apply_move(move, PlayerOrder((turn+1)%2)): return move #Use future turn for board update
            elif self.print_gamestates: print("Illegal Turn. Try again.")

        tui.terminate_game()

    def _play_the_game(self) -> int:
        """The actual hexapawn game.
        Players are requested to make moves.
        Moves are checked for legality.
        Moves are applied.
        Checked if a player wins the game.
        Repeated until a player has won.

        Returns:
            int: index of the player from players attribute
        """
        turn = 0
        while True:
            turn += 1

            move = self._player_turn(self.players[turn%2], turn)
            if move == False: return PlayerOrder((turn+1)%2) # When surrendering, the opposit player wins, thus turn +1.

            winning_player = self.board.check_win(PlayerOrder(turn%2))

            if self.print_gamestates:
                tui.print_gamestate(self.players[turn%2], turn, move, self.board)

            if winning_player != PlayerOrder.NOPLAYER: return winning_player

    def game(self):
        """Starts the game including preparations."""

        self.board.initialize_board()

        if self.print_gamestates:
            tui.print_startstate(self.player_one, self.player_two, self.board)

        winning_player = self._play_the_game()

        self._eval_game(winning_player)

