from abc import ABC
import numpy as np

from src.movement import Movement
import src.coordinate_functions as util
from src.enums import PlayerOrder

class BasePiece(ABC):
    """Abstract BaseClass for Hexapawn pieces."""

    piece_row: int
    piece_col: int
    player: int

    def __init__(self, piece_row: int, piece_col: int, player: int): ...

    def get_coordinates(self) -> tuple[int, int]: ...

    def get_all_moves(self, max_coordinate: int, board: np.ndarray
                      ) -> list[Movement]: ...

    def get_id(self) -> str: ...

    def move_piece(self, move: Movement):
        self.piece_row = move.end[0]
        self.piece_col = move.end[-1]

class Pawn(BasePiece):
    """Pawn class for Pawn to be used in Hexpawn."""
    direction: int

    def __init__(self, piece_row: int, piece_col: int, player: PlayerOrder):
        self.piece_row = piece_row
        self.piece_col = piece_col

        self.player = player

        if self.player == PlayerOrder.PLAYER1: self.direction = 1
        else: self.direction = -1

    def get_coordinates(self) -> tuple[int, int]:
        """Return current position of the Pawn instance.

        Returns:
            tuple[int, int]: Returns Row, Column
        """
        return self.piece_row, self.piece_col

    def get_coord_push(self, board: np.ndarray) -> list[Movement]:
        """Get coordinates this Pawn can be pushed to.
        Pushing can not capture pieces and does only go forward in facing direction.

        Args:
            board (np.ndarray): Numpy Array representation of a board.

        Returns:
            list[Movement]: Returns all possible Moves in list.
                            If no possible move, returns empty list.
        """

        end_point = (self.piece_row + self.direction, self.piece_col)

        if not util.coordinates_in_range(end_point, len(board)-1): return []

        if board[end_point]: return [] # Board field must be empty

        return [Movement(
            self.piece_row, self.piece_col,
            self.piece_row + self.direction, self.piece_col
            )]


    def get_coord_capture(self, board: np.ndarray) -> list[Movement]:
        """Get coordinates the Pawn can capture.
        Capturing can only apply diagonally ind facing direction and requires enemy piece on this coordinate.

        Args:
            board (np.ndarray): Numpy Array representation of a board.

        Returns:
            list[Movement]: Returns all possible Moves in list.
                            If no possible move, returns empty list.
        """
        out = []

        end_points = [
            (self.piece_row + self.direction, self.piece_col + 1),
            (self.piece_row + self.direction, self.piece_col - 1)
        ]

        for end_point in end_points:
            if not util.coordinates_in_range(end_point, len(board)-1): continue
            if not board[end_point]: continue
            if board[end_point].player == self.player: continue

            out.append(Movement(self.piece_row, self.piece_col,
                        end_point[0], end_point[1]))

        return out

    def get_all_moves(self, board: np.ndarray) -> list[Movement]:
        return self.get_coord_push(board) + self.get_coord_capture(board)

    def get_id_unique(self) -> str:
        """Get unique ID based on coordinates and player."""
        return f"{self.piece_row}{self.piece_col}P{int(self.player)}"

    def get_id(self) -> str:
        """Get simple ID based on Player. ID is not unique"""
        return f"P{int(self.player)}"

    def __str__(self) -> str:
        """Return String representation based on Player."""
        if self.player == PlayerOrder.PLAYER1:
            return "\033[32m" + "P" + "\033[0m" # Print in Green
        else:
            return "\033[31m" + "P" + "\033[0m" # Print in Red
