# Classes and functions to play the game Shreksapawn
# Code written by Berenike Blaser on 06 July 2023

# Comments added by Georg Veh on 18 July 2023

####################################################################################

### Import external modules

# First, we import modules that we will need to define special classes (dataclass), 
# make a random move, and set a timer.

from dataclasses import dataclass, field 
from abc import ABC, abstractmethod 
import random
import time

####################################################################################

### Define the 'user' that plays against the computer. Overall, the code allows the 
# user to interact with the game by inputting their name and making moves using 
# available pieces on the board. It validates the user's input and performs 
# corresponding actions based on the chosen piece and move.

# We use the decorator 'dataclass'. The dataclass() decorator examines the class to 
# find 'fields'. A 'field' is defined as a class variable that has a type annotation.
# The @dataclass decorator is used to automatically generate common methods and 
# boilerplate code for the class based on its defined fields. In this case, it is used 
# to create the User class.

@dataclass
class User:

    # The User class has a single attribute called name, which is initialized to the 
    # default value "User" when an instance of the class is created. The name attribute 
    # is of type str.
    name: str = "User" 

    # The putName method allows the user to input their name from the console. It 
    # prompts the user to provide their name using the input() function and assigns 
    # the input value to the name attribute of the object.
    def putName(self):
        self.name = input("Please provide your name: ")
    
    # Then the user will be asked to make a move using the 'makeMove method'. The
    # single parameter in that method is 'board'.
    def makeMove(self, board):

        # A list called piecesOnBoard is created using a list comprehension. A list 
        # comprehension creates a new list by iterating over an existing iterable 
        # (such as a list, tuple, string, or range) and applying an expression or 
        # condition to each element. It provides a more compact syntax compared to 
        # traditional 'for' loops when constructing lists. Here the the pieces on the 
        # are filtered board based on two conditions: 
        # the piece must be an instance of UserPiece and there must be at least one 
        # possible move for that piece according to the board.whichMovesPossible() method.
        # Both UserPiece and whichMovesPossible have not been defined so far.

        piecesOnBoard = [piece for piece in board.pieces if (isinstance(piece, UserPiece) and bool(board.whichMovesPossible(piece)))]

        # The method prompts the user to select a piece to move. It generates a string 
        # 'promptPiece' containing the index of each eligible piece along with its 
        # position. 

        promptPiece = "" # Initializes an empty string to store the final constructed prompt.

        # The following 'for' loop iterates over the range of indices from 0 to 
        # len(piecesOnBoard) - 1. By the end of the loop, the promptPiece string will contain
        # a concatenation of all the pieces' information, each separated by two spaces. 
        # This string can be used as part of a prompt to display the available pieces and 
        # their positions on the board to the user.

        for i in range(len(piecesOnBoard)):
            promptPiece += str(i) + "-piece on " + piecesOnBoard[i].positionHorizontal + str(piecesOnBoard[i].positionVertical) + "  "

        # The while loop repeatedly prompts the user to choose a piece by entering its 
        # corresponding index, i.e. the location in a row on the board. 

        while True:
            i = input("Which piece would you like to move? " + promptPiece)

            # The user's input is validated to ensure it is a valid integer within the 
            # range of available pieces. 

            if i.isdigit() and int(i) in range(len(piecesOnBoard)):
                i = int(i)
                break
            else:
                print()
                print("Please answer with one of the given options (e.g. type '0').")
                print()

        chosenPiece = piecesOnBoard[i]

        # We then initialise a string 'promptMoves', and iterate over the possible moves 
        # for the chosenPiece. The string contains the index and description of each move.

        promptMoves = ""
        possibleMoves = board.whichMovesPossible(chosenPiece)
        for i in range(len(possibleMoves)):
            promptMoves += str(i) + "-" + possibleMoves[i] + "  "
        
        # Afterwards, we enter another while loop. The user is presented the potential moves
        # from the previous 'prompMoves'; the user is asked to select a move for 
        # the chosen piece by entering its corresponding index. Similar to the previous 
        # input validation, the while loop ensures the input is a valid integer within 
        # the range of available moves. The selected move is stored in the chosenMove 
        # variable.

        while True:
            j = input("Where would you like to move it to? " + promptMoves)
            if j.isdigit() and int(j) in range(len(piecesOnBoard)):
                j = int(j)
                break
            else:
                print()
                print("Please answer with one of the given options (e.g. type '0').")
                print()

        # Finally, the chosenPiece is moved to the selected move using the 
        # chosenPiece.move(chosenMove) method.

        chosenMove = possibleMoves[j]
        chosenPiece.move(chosenMove)

        # Finally, we iterate over the board.pieces list and checks if the chosen piece 
        # matches any SystemPiece objects. If a match is found, it removes the system piece 
        # from the board using the board.removePiece() method.
        
        for i in range(len(board.pieces)):
            if isinstance(board.pieces[i], SystemPiece) and chosenPiece == board.pieces[i]:
                board.removePiece(i)
                break

############################################################################################

# Create a board, on which the pieces will be set afterwards.
# The board is a dictonary with 3 entries A, B, and C representing the rows on the board. 
# Each row has 3 (so far empty) fields. 

def createStartingBoard():
    return {"A": [" "]*3, "B": [" "]*3, "C": [" "]*3}

############################################################################################

# Define the hexapawn board. The 'Board' class represents a board in a game and provides 
# methods to manage the state of the board, display it, check for game-ending conditions, 
# and remove pieces from the board.

# Parts of the 'Board' class depend on the implementation details of createStartingBoard(), 
# the UserPiece and SystemPiece classes, and other related functionality that are defined in
# their own classes and methods.

class Board:

    # First, we initialize the method of the 'Board' class. We define a parameter 'pieces'
    # and initialize the attributes of the 'Board' object. We set 'self.const' to the 
    # result of createStartingBoard(), and self.pieces to the value of the 'pieces' parameter.

    def __init__(self, pieces):
        self.const = createStartingBoard()
        self.pieces = pieces
    
    # The following method removes a piece from the self.pieces list. It takes an argument 
    # 'piece_num', which represents the index of the piece to be removed. The pop() method is 
    # used to remove the piece from the list.

    def removePiece(self, piece_num):
        self.pieces.pop(piece_num)

    # The method showBoard() prints the current state of the game board to the console. 
    # It shows the board by modifying self.const (a data structure representing the board) 
    # based on the positions and colors (in essence, it's not a color, but 'x' or 'o' as 
    # defined in the 'SystemPiece' and 'UserPiece' class) of the pieces in self.pieces. It 
    # then prints the board using a header (the indices 0, 1, 2) and separator lines.

    def showBoard(self):
        self.const = createStartingBoard()
        for piece in self.pieces:
            self.const[piece.positionHorizontal][piece.positionVertical] = piece.col
        print("  | 0 | 1 | 2")
        print("---------------")
        for key, x in self.const.items():
            print(key,"|", x[0],"|",x[1],"|",x[2])
            print("---------------")
        print()

    # The method whichMovesPossible() determines the possible moves for a given 
    # piece on the board. It takes a piece object as an argument and checks its 
    # position and the surrounding positions in self.const. 
    # Depending on the type of the piece (UserPiece or SystemPiece), it checks different 
    # conditions to determine which moves are possible. It returns a list of possible 
    # move options.
            
    def whichMovesPossible(self, piece):
        possibleMoves = [] # Initialize an empty list to store the possible moves for the piece.

        # If it's not found, a message is printed indicating that the piece is not on the board.

        if not piece in self.pieces:
            print("This piece is not on the board.")

        # We check if the piece is an instance of the UserPiece class. 
        # If it is, the following code block is executed to determine the possible moves for a 
        # user-controlled piece. The if statements evaluate different conditions to determine if
        # specific moves are possible for the piece based on its position and the 
        # state of the board.

        if isinstance(piece, UserPiece):
            if piece.positionVertical>0 and \
                ((piece.positionHorizontal == "C" and self.const["B"][piece.positionVertical-1] == "x") or \
                (piece.positionHorizontal == "B" and self.const["A"][piece.positionVertical-1] == "x")):
                possibleMoves.append("left")
            if (piece.positionHorizontal == "C" and self.const["B"][piece.positionVertical] == " ") or \
                (piece.positionHorizontal == "B" and self.const["A"][piece.positionVertical] == " "):
                possibleMoves.append("straight")
            if piece.positionVertical<2 and \
                ((piece.positionHorizontal == "C" and self.const["B"][piece.positionVertical+1] == "x") or \
                (piece.positionHorizontal == "B" and self.const["A"][piece.positionVertical+1] == "x")):
                possibleMoves.append("right")

        # The other if statements with isinstance(piece, SystemPiece) evaluate similar
        # conditions but for pieces of the SystemPiece class (system-controlled pieces). 
        # These conditions check for specific positions and neighboring pieces 
        # ("o" for system-controlled pieces).

        if isinstance(piece, SystemPiece):
            if piece.positionVertical>0 and \
                ((piece.positionHorizontal == "A" and self.const["B"][piece.positionVertical-1] == "o") or \
                (piece.positionHorizontal == "B" and self.const["C"][piece.positionVertical-1] == "o")):
                possibleMoves.append("left")
            if (piece.positionHorizontal == "A" and self.const["B"][piece.positionVertical] == " ") or \
                (piece.positionHorizontal == "B" and self.const["C"][piece.positionVertical] == " "):
                possibleMoves.append("straight")
            if piece.positionVertical<2 and \
                ((piece.positionHorizontal == "A" and self.const["B"][piece.positionVertical+1] == "o") or \
                (piece.positionHorizontal == "B" and self.const["C"][piece.positionVertical+1] == "o")):
                possibleMoves.append("right")
        return possibleMoves
    
    # This method checks if the game has reached its end. It examines the current state 
    # of the game by iterating over self.pieces. It categorizes the pieces as UserPiece 
    # or SystemPiece and checks their positions to see if any game-ending conditions 
    # have been met. If the game has ended, it prints a message and returns True. 
    # Otherwise, it returns False.
    
    def endGame(self):

        # We initialise three lists, which will be used to store information 
        # about the pieces on the board.

        movesPossible = [] 
        userPieces = []
        systemPieces = []

        # Inside the loop, it checks the type of each piece using isinstance(). 
        # If the piece is an instance of the UserPiece class, it appends it to
        #  the userPieces list. Additionally, if the piece's positionHorizontal 
        # attribute is equal to "A", it prints "Game Over" and immediately returns 
        # True, indicating that the game has ended.

        for piece in self.pieces:
            if isinstance(piece, UserPiece): # check if the piece is a user piece.
                userPieces.append(piece)

                if piece.positionHorizontal == "A":
                    print("Game Over")
                    return True
            
            # If the piece is an instance of the SystemPiece class, it appends it 
            # to the systemPieces list. Similarly, if the piece's positionHorizontal 
            # attribute is equal to "C", it prints "Game Over" and returns True.       

            elif isinstance(piece, SystemPiece):
                systemPieces.append(piece)
                if piece.positionHorizontal == "C":
                    print("Game Over")
                    return True

            # After processing each piece, we append the result of 
            # bool(self.whichMovesPossible(piece)) to the movesPossible list. This evaluates 
            # whether there are any possible moves for the current piece.

            movesPossible.append(bool(self.whichMovesPossible(piece)))

        #  Finally, we perform several checks to determine if the game has ended.

        # If either userPieces or systemPieces lists are empty (i.e., no user 
        # pieces or system pieces are left), it prints "Game Over" and returns True.

        if (not bool(userPieces)) or (not bool(systemPieces)):
            print("Game Over")
            return True
        
        # If there are no possible moves for any piece on the board, it prints 
        # "Game Over" and returns True.

        if not any(movesPossible):
            print("Game Over")
            return True
        
        # If none of the above conditions are met, it means the game is still in 
        # progress. In this case, the method returns False to indicate that the 
        # game has not ended yet. 

        return False

#########################################################################################

# Define the class 'Piece' as an abstract class. An abstract class is a class that cannot 
# be instantiated on its own and is meant to be subclassed by other classes. It serves as 
# a blueprint for creating derived classes that share a common interface or behavior. 
# Abstract classes provide a way to define common methods or attributes that subclasses 
# are expected to implement or override.

# In this case, the position of the 'Piece' is defined by the its arguments of a horizontal 
# and a vertical position. 
# Using super() allows us to work with inheritance and abstract classes.
# It ensures proper method resolution order and enables subclasses to extend or modify 
# the behavior of their parent classes while maintaining the inheritance hierarchy.

class Piece(ABC):
    def __init__(self, positionHorizontal, positionVertical):
        super().__init__()
        self.positionHorizontal = positionHorizontal
        self.positionVertical = positionVertical
    
        # By implementing the '__eq__' method, you can customize how instances of your 
        # class are compared for equality. This allows you to define equality based on 
        # specific attributes or conditions relevant to your class.

    def __eq__(self, value):

        # The isinstance() function in Python is used to check if an object belongs to 
        # a specific class or if it is an instance of a subclass. It takes two 
        # arguments: the object being tested and the class or tuple of classes to check 
        # against.

        if not isinstance(value, Piece):
            raise ValueError(f"{value} is not part of Piece class")
        return(self.positionHorizontal == value.positionHorizontal and
               self.positionVertical == value.positionVertical )

    def __str__(self):

        # The f"..." syntax is used to create f-strings, which are a way to format 
        # strings using expressions inside curly braces {}.
        # The f"..." syntax is typically used when you want to create a string that 
        # includes the values of variables or expressions within it. It allows you to 
        # embed the values directly into the string without the need for additional 
        # string concatenation or formatting.

        return f"{self.col} in {self.positionHorizontal}"+ str(self.positionVertical)

    def move(self):
        pass

####################################################################################

# Define a class for the piece whose position the user is asked to change.
# The UserPiece class inherits from the base class 'Piece', i.e. it is initialised
# with the same methods.

class UserPiece(Piece): # inherits from 'Piece' class

    # Overrides the __init__ method to initialize the positionHorizontal and 
    # positionVertical attributes by calling the parent class's __init__ method 
    # using super().__init__(positionHorizontal, positionVertical).

    def __init__(self, positionHorizontal, positionVertical):
        super().__init__(positionHorizontal, positionVertical)
        self.col = "o" # piece is shown as an 'o' on the board.

    # We define a move method that determines how the user-controlled piece moves based 
    # on the given direction. It updates the positionHorizontal attribute based on the 
    # current value and the specified rules. It also updates the positionVertical 
    # attribute based on the given direction.

    def move(self, direction):
        if self.positionHorizontal == "C":
            self.positionHorizontal = "B"
        elif self.positionHorizontal == "B":
            self.positionHorizontal = "A"
        if direction == "left":
            self.positionVertical -= 1
        if direction == "right":
            self.positionVertical += 1


# How the piece on the system side moves is mirrored to the user's piece.

class SystemPiece(Piece):
    def __init__(self,positionHorizontal, positionVertical):
        super().__init__(positionHorizontal, positionVertical)
        self.col = "x" # The objects are displayed as an 'x' on the board.
    def move(self, direction):
        if self.positionHorizontal == "A":
            self.positionHorizontal = "B"
        elif self.positionHorizontal == "B":
            self.positionHorizontal = "C"
        if direction == "left":
            self.positionVertical -= 1
        if direction == "right":
            self.positionVertical += 1

# The Strategy class represents a strategy for playing Shreksapawn based on 
# predefined board configurations and moves. The class determines the 
# current board configuration, selects a random move for that configuration, 
# and executes the move on the board. It also provides the ability to remove 
# losing moves from consideration.

class Strategy:
    def __init__(self):

        # First, we create a list called consts, which contains multiple 
        # dictionaries representing different board configurations. Each 
        # dictionary in consts represents a specific board configuration 
        # with positions and possible moves for each position.

        self.consts = [
            {"A": ["x", "x", "x"], "B": ["o", " ", " "], "C": [" ", "o", "o"]},
            {"A": ["x", "x", "x"], "B": [" ", " ", "o"], "C": ["o", "o", " "]},
            {"A": ["x", "x", "x"], "B": [" ", "o", " "], "C": ["o", " ", "o"]},
            {"A": ["x", " ", "x"], "B": ["x", "o", " "], "C": [" ", " ", "o"]},
            {"A": ["x", " ", "x"], "B": [" ", "o", "x"], "C": ["o", " ", " "]},
            {"A": [" ", "x", "x"], "B": ["o", "x", " "], "C": [" ", " ", "o"]},
            {"A": ["x", "x", " "], "B": [" ", "x", "o"], "C": ["o", " ", " "]},
            
            {"A": ["x", " ", "x"], "B": ["o", "o", " "], "C": [" ", "o", " "]},
            {"A": ["x", " ", "x"], "B": [" ", "o", "o"], "C": [" ", "o", " "]},
            {"A": ["x", "x", " "], "B": ["o", " ", "o"], "C": [" ", " ", "o"]},
            {"A": [" ", "x", "x"], "B": ["o", " ", "o"], "C": ["o", " ", " "]},
            {"A": [" ", "x", "x"], "B": [" ", "x", "o"], "C": ["o", " ", " "]},
            {"A": ["x", "x", " "], "B": ["o", "x", " "], "C": [" ", " ", "o"]},
            {"A": [" ", "x", "x"], "B": ["x", "o", "o"], "C": ["o", " ", " "]},
            {"A": ["x", "x", " "], "B": ["o", "o", "x"], "C": [" ", " ", "o"]},

            {"A": ["x", " ", "x"], "B": ["x", " ", "o"], "C": [" ", "o", " "]},
            {"A": ["x", " ", "x"], "B": ["o", " ", "x"], "C": [" ", "o", " "]},
            {"A": [" ", "x", "x"], "B": [" ", "o", " "], "C": [" ", " ", "o"]},
            {"A": ["x", "x", " "], "B": [" ", "o", " "], "C": ["o", " ", " "]},
            {"A": [" ", "x", "x"], "B": [" ", "o", " "], "C": ["o", " ", " "]},
            {"A": ["x", "x", " "], "B": [" ", "o", " "], "C": [" ", " ", "o"]},

            {"A": ["x", " ", "x"], "B": ["o", " ", " "], "C": [" ", " ", "o"]},
            {"A": ["x", " ", "x"], "B": [" ", " ", "o"], "C": ["o", " ", " "]},
            {"A": [" ", " ", "x"], "B": ["x", "x", "o"], "C": [" ", " ", " "]},
            {"A": ["x", " ", " "], "B": ["o", "x", "x"], "C": [" ", " ", " "]},
            {"A": [" ", " ", "x"], "B": ["o", "o", "o"], "C": [" ", " ", " "]},
            {"A": ["x", " ", " "], "B": ["o", "o", "o"], "C": [" ", " ", " "]},
            {"A": [" ", "x", " "], "B": ["x", "o", "o"], "C": [" ", " ", " "]},
            {"A": [" ", "x", " "], "B": ["o", "o", "x"], "C": [" ", " ", " "]},

            {"A": ["x", " ", " "], "B": ["x", "x", "o"], "C": [" ", " ", " "]},
            {"A": [" ", " ", "x"], "B": ["o", "x", "x"], "C": [" ", " ", " "]},
            {"A": ["x", " ", " "], "B": [" ", "o", "x"], "C": [" ", " ", " "]},
            {"A": [" ", " ", "x"], "B": ["x", "o", " "], "C": [" ", " ", " "]},

            {"A": [" ", "x", " "], "B": [" ", "x", "o"], "C": [" ", " ", " "]},
            {"A": [" ", "x", " "], "B": ["x", "o", " "], "C": [" ", " ", " "]},
            {"A": ["x", " ", " "], "B": ["x", "o", " "], "C": [" ", " ", " "]},
            {"A": [" ", " ", "x"], "B": [" ", "o", "x"], "C": [" ", " ", " "]},
        ]

        self.allmoves = [] # initialise an empty list called 'allmoves'.

        # In this loop, each board configuration (const) stored in the self.consts 
        # list is processed. For each configuration, the loop iterates over its 
        # key-value pairs using const.items(). Each key represents a column name, 
        # and each value represents a row containing the positions of the pieces.
        # The allmoves list is then used by other methods in the Strategy class 
        # to make moves based on the predefined strategies.

        for const in self.consts:

            # initialise an empty list of piece positions and directions of potential moves.

            const_options = [] 
            for column, row in const.items():

                # Within the inner loop, we iterate over each element (row[i]) 
                # in the current row. If the element is equal to "x", it means 
                # there is a possible piece at that position.

                for i in range(len(row)):
                    positionHorizontal = column
                    positionVertical = i
                    if row[i] == "x":
                        possiblePiece = [positionHorizontal, positionVertical] # The position of the piece
                        possibleMoves = [] # Empty list containing the direction of possible move

                        # We check various conditions to determine the available moves for the piece

                        if positionVertical>0 and \
                            ((positionHorizontal == "A" and const["B"][positionVertical-1] == "o") or \
                            (positionHorizontal == "B" and const["C"][positionVertical-1] == "o")):
                            possibleMoves.append("left")
                        if (positionHorizontal == "A" and const["B"][positionVertical] == " ") or \
                            (positionHorizontal == "B" and const["C"][positionVertical] == " "):
                            possibleMoves.append("straight")
                        if positionVertical<2 and \
                            ((positionHorizontal == "A" and const["B"][positionVertical+1] == "o") or \
                            (positionHorizontal == "B" and const["C"][positionVertical+1] == "o")):
                            possibleMoves.append("right")
                        if bool(possibleMoves):
                            for move in possibleMoves:
                                const_options.append((positionHorizontal,positionVertical,move))

            # After processing all positions in the current board configuration, a list 
            # containing the configuration itself (const) and the corresponding options 
            # (const_options) is appended to the allmoves list. This list represents all 
            # the available moves for the current board configuration.

            self.allmoves.append([const, const_options])

            # The process is repeated for each board configuration in self.consts, resulting in the 
            # allmoves list containing all possible moves for each configuration.

    # Next, we define a method that takes a board object as input and compares its 
    # 'const' attribute (representing the current board configuration) with the 
    # configurations stored in the consts list. It loops through the 'consts' list and 
    # checks if any configuration matches the board's 'const'. 
    # If a match is found, it returns the index of the matching configuration in the consts list.

    def recognizeConst(self,board):
        for i in range(len(self.consts)):
            if self.consts[i] == board.const:
                return(i)

    # Then we make a random move by taking index i as input, which represents the index of 
    # a specific board configuration in the 'consts' list. The method selects a random move from 
    # the available moves for that configuration. It does so by accessing the 'const_options' list within the 
    # corresponding configuration in 'allmoves'. The selected move index is then returned.

    def pickRandomMove(self, i):
        randomOption = random.randrange(len(self.allmoves[i][1]))
        return randomOption
    
    # The method makeMove() makes a move based on the strategy. It takes a board object
    # as input. First, it determines the current board configuration by calling the 
    # recognizeConst(board) method. It retrieves the index of the matching configuration 
    # in the consts list.
    # Next, it selects a random move for that configuration by calling pickRandomMove(whichConst). 
    # The whichConst variable holds the index of the current board configuration. The 
    # method retrieves the corresponding list of moves from allmoves using the whichConst 
    # index, and selects a random move from that list.
    # The selected move is then printed, simulating the system's decision-making process. 
    # A delay of 2 seconds is introduced using time.sleep(2) for visual purposes.
    # Finally, the method finds the piece on the board that matches the selected move. 
    # It iterates through the board.pieces list and checks if a piece's position matches 
    # the selected move's position. If a match is found, it calls the piece's move() 
    # method with the selected move's direction as an argument. If the matched piece is a 
    # user piece, it is removed from the board using board.removePiece(k).
    
    def makeMove(self, board):

        # We call the recognizeConst method of the Strategy class, passing the board 
        # object as an argument. This method determines the current board configuration 
        # by comparing the const attribute of the board with the configurations stored 
        # in the consts list. The index of the matching configuration is assigned to the 
        # whichConst variable.

        whichConst = self.recognizeConst(board)

        # We call the pickRandomMove method of the Strategy class, passing the whichConst 
        # variable as an argument. This method selects a random move index from the available
        # moves for the corresponding board configuration. The selected move index is 
        # assigned to the whichMove variable.

        whichMove = self.pickRandomMove(whichConst)

        # We then retrieve the selected move based on the whichConst and whichMove indices. 
        # The allmoves list contains all the available moves for each board configuration. 
        # The [1] index accesses the list of moves for the specific board configuration, 
        # and the whichMove index retrieves the selected move tuple. The selectedMove variable 
        # now holds the position and direction of the system's chosen move.

        selectedMove = self.allmoves[whichConst][1][whichMove]

        # A message appears stating the system's chosen move. It displays the position information 
        # (selectedMove[0] and selectedMove[1]) with the direction (selectedMove[2]) to 
        # provide a clear description of the chosen move.

        print("The system choses the piece on " + selectedMove[0] + str(selectedMove[1]) + " to go " + selectedMove[2] + ".")
        time.sleep(2)

        # We iterate over all pieces on the board

        for piece in board.pieces:

            # We check if the piece's position matches the selected move's position and if the 
            # piece is an instance of the SystemPiece class. This condition ensures that the 
            # system can only move its own pieces, not user pieces.

            if piece.positionHorizontal == selectedMove[0] and \
            piece.positionVertical == selectedMove[1] and \
            isinstance(piece, SystemPiece):
                
                # If that is true, we move on the selected piece using the direction
                # specified in selectedMove[2]

                piece.move(selectedMove[2])

                # Then we iterate over the pieces on the board.

                for k in range(len(board.pieces)):
                    if isinstance(board.pieces[k], UserPiece) and \
                    board.pieces[k] == piece:
                        board.removePiece(k)
                        return [whichConst, whichMove]
                    
                # If the loop finishes without returning, it means that no user piece 
                # was found at the selected move position. In this case, [whichConst, whichMove] 
                # is still returned as a list.

                return [whichConst, whichMove]

    # This method removes a losing move from the allmoves list. It takes the 
    # indices whichConst and whichMove as input, representing the indices 
    # of the selected move within the allmoves list. It uses these indices 
    # to access the corresponding move and removes it from the list.

    def removeLosingMove(self, whichConst, whichMove):
        self.allmoves[whichConst][1].pop(whichMove)


# Finally, the Counter class records the game statistics and displaying them to the user. 

class Counter:

    # We initialize two attributes, totGames and wonByUser, to keep track of the total number 
    # of games played, and to record the number of games won by the user. 
    
    def __init__(self):
        self.totGames = 0 # Initial value set to zero in both cases
        self.wonByUser = 0

    # The displayWhoWon method shows the winner of the game and updates the game statistics. 
    # It takes two arguments: winner, which represents the winner of the game 
    # ("u" for the user or any other value for the system), and user, which is an instance 
    # of the User class representing the user playing the game.

    def displayWhoWon(self, winner, user):
        self.totGames += 1
        if winner == "u":
            self.wonByUser +=1
            print(user.name + " won!")
            print()
        else:
            print("The system won!")
            print()
    
    # Finally, we print a message using user.name to display the user's name, 
    # along with the number of games won by the user (self.wonByUser) a
    # nd the total number of games played (self.totGames).

    def displayCount(self, user):
        print("So far, " + user.name + " has won " + str(self.wonByUser) + " out of " + str(self.totGames) +" times.")