from string import ascii_uppercase
from typing import Iterable

def valid_int_input(input: str, allowed_input: Iterable[int] = []) -> bool:
    """Checks if input can be converted into int and if it is in a given Iterable of ints.

    Args:
        input (str): input to check
        allowed_input (Iterable[int], optional): Int values allowed. Empty Iterable ignores this condition. Defaults to [].

    Returns:
        bool: Whether check was successful or not.
    """
    try: input_int = int(input)
    except ValueError: return False

    if input_int in allowed_input or not allowed_input: return True
    else: return False


def coordinates_in_range(point: tuple[int, int], max_val: int) -> bool:
    """Checks if all values of a given tuple are >= 0 and <= to any given number.

    Args:
        point (tuple[int, int]): Point with coordinates to check
        max_val (int): Maximum value allowed

    Returns:
        bool: True if all values are in range or False if at least one is not.
    """
    for val in point:
        if val < 0 or val > max_val: return False
    return True

def matrix_to_coord(row: int, col: int) -> str:
    """Get the coordinate format for a board field (columns are letters) from given matric presentation.

    Args:
        row (int): Row index of Array presentation
        col (int): Column index of Array presentation

    Returns:
        str: coordinate presentation of board field. Format: [a-zA-Z][0-9]
    """
    return f"{ascii_uppercase[col]}{row}"

def coord_to_matrix(coordinate: str) -> tuple[int, int]:
    """Converts fiven coordinate presentation of board field to matrix presentation.
    Letters are converted using ASCII decoding.

    Args:
        move (str): Coordinate presentation of board field. Format: [a-zA-Z][0-9]

    Returns:
        tuple[int, int]: tuple of int row, column values
    """
    return int(coordinate[1]), ord(coordinate[0].lower()) - 97
